<?php

namespace Modules\Payroll\Database\Seeders;

use Exception;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Modules\Academic\Models\Staff;
use Modules\Academic\Models\Teacher;
use Modules\Payroll\Models\PayslipInvoice;
use Modules\Payroll\Models\PayslipSalary;
use Modules\Payroll\Models\PayslipSalaryHead;
use Modules\Payroll\Models\SalaryHead;

class PayslipSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $months = [10, 11, 12]; // Oct–Dec
        $year = 2025;

        DB::beginTransaction();
        try {
            // Collect all staff and teacher user_ids
            $allUsers = collect();

            $staffData = Staff::all();
            foreach ($staffData as $staff) {
                if ($staff && $staff->user_id) {
                    $allUsers->push($staff->user_id);
                }
            }

            $teacherData = Teacher::all();
            foreach ($teacherData as $teacher) {
                if ($teacher && $teacher->user_id) {
                    $allUsers->push($teacher->user_id);
                }
            }

            // Fetch all salary heads once
            $salaryHeads = SalaryHead::all();

            foreach ($allUsers as $userId) {
                foreach ($months as $month) {
                    // Skip if already exists
                    $exists = PayslipSalary::where('user_id', $userId)
                        ->where('year', $year)
                        ->where('month', $month)
                        ->first();

                    if ($exists) {
                        continue;
                    }

                    // Create PayslipSalary
                    $payslipSalary = new PayslipSalary([
                        'institute_id' => 1,
                        'branch_id' => 1,
                        'user_id' => $userId,
                        'year' => $year,
                        'month' => $month,
                    ]);
                    $payslipSalary->save();

                    // Attach salary heads
                    foreach ($salaryHeads as $salaryHead) {
                        PayslipSalaryHead::create([
                            'institute_id' => 1,
                            'branch_id' => 1,
                            'user_payroll_id' => $payslipSalary->id,
                            'salary_head_id' => $salaryHead->id,
                            'amount' => 100, // demo random amount
                        ]);
                    }

                    // Create PayslipInvoice
                    PayslipInvoice::create([
                        'institute_id' => 1,
                        'branch_id' => 1,
                        'invoice_id' => $payslipSalary->id.rand(100000, 999999),
                        'payslip_salary_id' => $payslipSalary->id,
                        'created_at' => now(),
                    ]);
                }
            }

            DB::commit();
            $this->command->info('✅ Demo Payslips generated for Staff & Teachers (Oct–Dec 2025)');
        } catch (Exception $e) {
            DB::rollback();
            $this->command->error('❌ Payslip generation failed: '.$e->getMessage());
        }
    }
}
