<?php

namespace Modules\Frontend\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Traits\RequestSanitizerTrait;
use Exception;
use Illuminate\Http\JsonResponse;
use Modules\Frontend\Http\Requests\AcademicImage\AcademicImageStoreRequest;
use Modules\Frontend\Http\Requests\AcademicImage\AcademicImageUpdateRequest;
use Modules\Frontend\Repositories\AcademicImageRepository;
use OpenApi\Annotations as OA;

class AcademicImageController extends Controller
{
    use RequestSanitizerTrait;

    public function __construct(private AcademicImageRepository $academicImage) {}

    /**
     * @OA\Get(
     *     path="/api/v1/academic-images",
     *     tags={"Academic Images - LMS"},
     *     summary="Get all academic-images for REST API",
     *
     *     @OA\Parameter(name="perPage", in="query", description="Per page count", required=false, explode=true, @OA\Schema(default="10", type="integer")),
     *     security={{"bearer":{}}},
     *
     *     @OA\Response(response=200, description="Successful data fetch"),
     *     @OA\Response(response=400, description="Invalid filter data"),
     *     @OA\Response(response=500, description="Internal server error")
     * )
     */
    public function index(): JsonResponse
    {
        try {
            return $this->responseSuccess(
                $this->academicImage->getAll(request()->all()),
                'AcademicImages has been fetched successfully.'
            );
        } catch (Exception $e) {
            return $this->responseError([], $e->getMessage());
        }
    }

    /**
     * @OA\Post(
     *     path="/api/v1/academic-images",
     *     tags={"Academic Images - LMS"},
     *     summary="Create a new academicImage",
     *     security={{"bearer":{}}},
     *
     *     @OA\RequestBody(
     *         description="AcademicImages object",
     *         required=true,
     *
     *         @OA\MediaType(
     *            mediaType="application/json",
     *
     *            @OA\Schema(
     *                 type="object",
     *
     *                 @OA\Property(property="title", description="AcademicImages Title", type="string", example="AcademicImages "),
     *                 @OA\Property(property="answer", description="AcademicImages Subtitle", type="string", example="AcademicImages Answer"),
     *                 @OA\Property(property="status", description="AcademicImages Status", type="integer", example=1),
     *                 required={"title", "answer", "status"}
     *             )
     *         ),
     *     ),
     *
     *     @OA\Response(response=200, description="AcademicImages created successfully"),
     *     @OA\Response(response=400, description="Invalid academicImage data"),
     *     @OA\Response(response=403, description="Unauthorized to add academicImage"),
     *     @OA\Response(response=500, description="Internal server error")
     * )
     */
    public function store(AcademicImageStoreRequest $request): JsonResponse
    {
        try {
            return $this->responseSuccess(
                $this->academicImage->create($request->all()),
                'AcademicImages has been created successfully.'
            );
        } catch (Exception $exception) {
            return $this->responseError([], $exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * @OA\Get(
     *     path="/api/v1/academic-images/{id}",
     *     tags={"Academic Images - LMS"},
     *     summary="Show academicImage",
     *     security={{"bearer":{}}},
     *
     *     @OA\Parameter(name="id", description="AcademicImages ID", required=true, @OA\Schema(type="integer"), in="path", example=1),
     *
     *     @OA\Response(response=200, description="Successful AcademicImages find"),
     *     @OA\Response(response=403, description="Unauthorized to delete AcademicImages"),
     *     @OA\Response(response=404, description="AcademicImages not found"),
     *     @OA\Response(response=500, description="Internal server error")
     * )
     */
    public function show(int $id): JsonResponse
    {
        try {
            return $this->responseSuccess(
                $this->academicImage->getById($id),
                'AcademicImages has been fetched successfully.'
            );
        } catch (Exception $exception) {
            return $this->responseError([], $exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * @OA\Post(
     *     path="/api/v1/academic-images/{id}",
     *     tags={"Academic Images - LMS"},
     *     summary="Update an existing academicImage",
     *     security={{"bearer":{}}},
     *
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="AcademicImages ID",
     *
     *         @OA\Schema(type="integer"),
     *         example=1
     *     ),
     *
     *     @OA\Parameter(
     *         name="_method",
     *         in="query",
     *         required=true,
     *         description="Method (should be 'PUT' for updates)",
     *
     *         @OA\Schema(type="string", default="PUT")
     *     ),
     *
     *     @OA\RequestBody(
     *         description="AcademicImages object",
     *         required=true,
     *
     *         @OA\MediaType(
     *            mediaType="application/json",
     *
     *            @OA\Schema(
     *                 type="object",
     *
     *                 @OA\Property(property="title", description="AcademicImages Title", type="string", example="AcademicImages "),
     *                 @OA\Property(property="answer", description="AcademicImages Subtitle", type="string", example="AcademicImages Answer"),
     *                 @OA\Property(property="status", description="AcademicImages Status", type="integer", example=1),
     *                 required={"academicImage_title", "title", "answer", "status"}
     *             )
     *         )
     *     ),
     *
     *     @OA\Response(response=200, description="AcademicImages updated successfully"),
     *     @OA\Response(response=400, description="Invalid academicImage data"),
     *     @OA\Response(response=405, description="Method not allowed"),
     *     @OA\Response(response=403, description="Unauthorized to update academicImage"),
     *     @OA\Response(response=404, description="AcademicImages not found"),
     *     @OA\Response(response=500, description="Internal server error")
     * )
     */
    public function update(AcademicImageUpdateRequest $request, int $id): JsonResponse
    {
        try {
            return $this->responseSuccess(
                $this->academicImage->update($id, $this->getUpdateRequest($request)),
                'AcademicImages has been updated successfully.'
            );
        } catch (Exception $exception) {
            return $this->responseError([], $exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * @OA\Delete(
     *     path="/api/v1/academic-images/{id}",
     *     tags={"Academic Images - LMS"},
     *     summary="Delete AcademicImages",
     *     security={{"bearer":{}}},
     *
     *     @OA\Parameter(name="id", description="AcademicImages ID", required=true, @OA\Schema(type="integer"), in="path", example=1),
     *
     *     @OA\Response(response=200, description="Successful academicImage delete"),
     *     @OA\Response(response=403, description="Unauthorized to delete academicImage"),
     *     @OA\Response(response=500, description="Internal server error")
     * )
     */
    public function destroy(int $id): JsonResponse
    {
        try {
            return $this->responseSuccess(
                $this->academicImage->delete($id),
                'AcademicImages has been deleted successfully.'
            );
        } catch (Exception $exception) {
            return $this->responseError([], $exception->getMessage(), $exception->getCode());
        }
    }
}
